<?php

namespace _ReCodingStandard\Sniffs\Enums;

use _ReCodingStandard\Support\TokenFinder;
use PHP_CodeSniffer\Files\File;
use PHP_CodeSniffer\Sniffs\Sniff;

final class EnumCasePascalCaseSniff implements Sniff
{
    public const CODE_ENUM_CASE_NOT_PASCAL_CASE = 'EnumCaseNotPascalCase';

    /**
     * Registers the tokens that this sniff wants to listen for.
     *
     * @return (int|string)[]
     */
    public function register(): array
    {
        return [
            T_ENUM_CASE,
        ];
    }

    /**
     * Called when one of the token types that this sniff is listening for
     * is found.
     *
     * @param \PHP_CodeSniffer\Files\File $phpcsFile
     * @param int $stackPtr
     * @return void
     */
    public function process(File $phpcsFile, $stackPtr): void
    {
        $tokens = $phpcsFile->getTokens();

        $caseNamePointer = TokenFinder::nextExceptEmpty($phpcsFile, $stackPtr + 1);

        if (is_null($caseNamePointer)) {
            return;
        }

        $caseName = $tokens[$caseNamePointer]['content'];

        if (!$this->stringIsLikelyPascalCase($caseName)) {
            $phpcsFile->addError(
                sprintf('Enum cases must be PascalCase; found %s', $caseName),
                $stackPtr,
                self::CODE_ENUM_CASE_NOT_PASCAL_CASE,
            );
        }
    }

    /**
     * Determine if the given string is likely to be PascalCase.
     *
     * @param string $str
     * @return bool
     */
    private function stringIsLikelyPascalCase(string $str): bool
    {
        if ($this->isLowerCase($str)) {
            return false;
        }

        if ($this->isUpperCase($str) && strlen($str) > 1) {
            // Check string length as well to allow single-letter cases in uppercase
            return false;
        }

        $firstCharacter = substr($str, 0, 1);

        if (!$this->isUpperCase($firstCharacter)) {
            return false;
        }

        $lastCharacter = substr($str, -1);

        if (!$this->isLowerCase($lastCharacter) && strlen($str) > 1) {
            // Check string length as well to allow single-letter cases in uppercase
            return false;
        }

        return true;
    }

    /**
     * Determine if the given string is lowercase.
     *
     * @param string $str
     * @return bool
     */
    private function isLowerCase(string $str): bool
    {
        return $str === strtolower($str);
    }

    /**
     * Determine if the given string is uppercase.
     *
     * @param string $str
     * @return bool
     */
    private function isUpperCase(string $str): bool
    {
        return $str === strtoupper($str);
    }
}
