<?php

namespace Recoded\TestingTools\Concerns;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Enumerable;
use PHPUnit\Framework\Assert;
use PHPUnit\Framework\Constraint\LogicalNot;
use Recoded\TestingTools\PHPUnit\Constraints\EnumerableContainsModel;
use Recoded\TestingTools\PHPUnit\Constraints\IsModel;

trait InteractsWithModels
{
    /**
     * Assert that the given enumerable contains the given model.
     *
     * @param \Illuminate\Database\Eloquent\Model $needle
     * @param \Illuminate\Support\Enumerable<array-key, mixed> $haystack
     * @param string $message
     * @return $this
     */
    protected function assertContainsModel(Model $needle, Enumerable $haystack, string $message = ''): static
    {
        $constraint = new EnumerableContainsModel($needle);

        Assert::assertThat($haystack, $constraint, $message);

        return $this;
    }

    /**
     * Assert that the given models refer to the same data in the database.
     *
     * @param \Illuminate\Database\Eloquent\Model $expected
     * @param mixed $actual
     * @param string $message
     * @return $this
     */
    protected function assertIsModel(Model $expected, mixed $actual, string $message = ''): static
    {
        Assert::assertThat($actual, new IsModel($expected), $message);

        return $this;
    }

    /**
     * Assert that the given models does not refer to the same data in the database.
     *
     * @param \Illuminate\Database\Eloquent\Model $expected
     * @param mixed $actual
     * @param string $message
     * @return $this
     */
    protected function assertIsNotModel(Model $expected, mixed $actual, string $message = ''): static
    {
        $constraint = new LogicalNot(
            new IsModel($expected),
        );

        Assert::assertThat($actual, $constraint, $message);

        return $this;
    }

    /**
     * Assert that the given enumerable does not contain the given model.
     *
     * @param \Illuminate\Database\Eloquent\Model $needle
     * @param \Illuminate\Support\Enumerable<array-key, mixed> $haystack
     * @param string $message
     * @return $this
     */
    protected function assertDoesNotContainModel(Model $needle, Enumerable $haystack, string $message = ''): static
    {
        $constraint = new LogicalNot(
            new EnumerableContainsModel($needle),
        );

        Assert::assertThat($haystack, $constraint, $message);

        return $this;
    }
}
