<?php

namespace Recoded\TestingTools\Http;

use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Illuminate\Support\Traits\Tappable;
use Illuminate\Testing\Assert as PHPUnit;
use JsonException;
use PHPUnit\Framework\Assert;
use Symfony\Component\HttpFoundation\HeaderBag;
use Symfony\Component\HttpFoundation\InputBag;

/**
 * @template T of \Illuminate\Foundation\Http\FormRequest
 */
final class TestFormRequest
{
    use Tappable;

    /**
     * The form request to delegate to.
     *
     * @var T
     */
    public FormRequest $baseRequest;

    /**
     * The (cached) validator for the request.
     *
     * @var \Illuminate\Contracts\Validation\Validator
     */
    private Validator $validator;

    /**
     * Create a new test form request instance.
     *
     * @param T $baseRequest
     * @return void
     * @internal This method is not covered by the backwards compatability promise.
     */
    public function __construct(FormRequest $baseRequest)
    {
        $this->baseRequest = $baseRequest;
    }

    /**
     * Set the user making the request.
     *
     * @param \Illuminate\Contracts\Auth\Authenticatable $user
     * @return $this
     */
    public function as(Authenticatable $user): self
    {
        $this->baseRequest->setUserResolver(static fn () => $user);

        return $this;
    }

    /**
     * Assert that the request fails authorization.
     *
     * @param string $message
     * @return $this
     */
    public function assertFailsAuthorization(string $message = ''): self
    {
        Assert::assertFalse(
            $this->passesAuthorization(),
            $message ?: 'The request authorized successfully',
        );

        return $this;
    }

    /**
     * Assert that the request fails validation.
     *
     * @param array<array-key, string|string[]>|string $expected
     * @param string $message
     * @return $this
     */
    public function assertFailsValidation(array|string $expected = [], string $message = ''): self
    {
        Assert::assertTrue(
            $this->getValidatorInstance()->fails(),
            $message ?: 'Validation passed unexpectedly',
        );

        if (empty($expected)) {
            return $this;
        }

        $failures = $this->getFailures();

        foreach (Arr::wrap($expected) as $key => $value) {
            PHPUnit::assertArrayHasKey(
                is_int($key) ? $value : $key,
                $failures,
                "Failed to find a validation error for key: '{$value}'" . PHP_EOL . $this->getFormattedFailures(),
            );

            if (!is_int($key)) {
                $hasError = false;

                foreach (Arr::wrap($failures[$key]) as $message) {
                    if (Str::contains($message, $value)) {
                        $hasError = true;

                        break;
                    }
                }

                if (!$hasError) {
                    PHPUnit::fail(
                        vsprintf("Failed to find a validation error for key and message: '%s' => '%s\r\n%s'", [
                            $key, $value, $this->getFormattedFailures(),
                        ]),
                    );
                }
            }
        }

        return $this;
    }

    /**
     * Assert that the request passes authorization.
     *
     * @param string $message
     * @return $this
     */
    public function assertPassesAuthorization(string $message = ''): self
    {
        Assert::assertTrue(
            $this->passesAuthorization(),
            $message ?: 'The request failed to authorize successfully',
        );

        return $this;
    }

    /**
     * Assert that the request passes validation.
     *
     * @param string[]|string $expected
     * @param string $message
     * @return $this
     */
    public function assertPassesValidation(array|string $expected = [], string $message = ''): self
    {
        if (empty($expected)) {
            $message = $message ?: 'Validation did not pass, the following failures occurred:';

            Assert::assertFalse(
                $this->getValidatorInstance()->fails(),
                $message . PHP_EOL . $this->getFormattedFailures(),
            );

            return $this;
        }

        $failures = $this->getFailures();

        foreach (Arr::wrap($expected) as $key) {
            PHPUnit::assertArrayNotHasKey(
                $key,
                $failures,
                "Found a validation error for key: '{$key}'" . PHP_EOL . $this->getFormattedFailures(),
            );
        }

        return $this;
    }

    /**
     * Get the validation failures.
     *
     * @return array<string, string[]>
     */
    protected function getFailures(): array
    {
        $validator = $this->getValidatorInstance();

        if (!$validator->fails()) {
            return [];
        }

        return $validator->errors()->toArray();
    }

    /**
     * Get the validation failures as a readable string.
     *
     * @return string
     */
    protected function getFormattedFailures(): string
    {
        try {
            return json_encode(
                $this->getFailures(),
                JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_THROW_ON_ERROR,
            );
        } catch (JsonException $e) {
            return sprintf('JSON exception: %s', $e->getMessage());
        }
    }

    /**
     * Get the underlying validator instance.
     *
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function getValidatorInstance(): Validator
    {
        return $this->validator ??= (function () {
            /** @var \Illuminate\Foundation\Http\FormRequest $this */
            $this->prepareForValidation();

            $validator = $this->getValidatorInstance();

            if (!$validator->fails()) {
                $this->passedValidation();
            }

            return $validator;
        })->call($this->baseRequest);
    }

    /**
     * Set the headers on the request.
     *
     * @param array<string, string|string[]> $headers
     * @return $this
     */
    public function headers(array $headers): self
    {
        $this->baseRequest->headers = new HeaderBag($headers);

        return $this;
    }

    /**
     * Set the input on the request.
     *
     * @param array<int|string, mixed> $input
     * @return $this
     */
    public function input(array $input): self
    {
        $this->baseRequest->request = new InputBag($input);

        return $this;
    }

    /**
     * Determine whether the request passes authorization.
     *
     * @return bool
     */
    protected function passesAuthorization(): bool
    {
        return (fn () => $this->passesAuthorization())->call($this->baseRequest);
    }

    /**
     * Prepare the data for validation.
     *
     * @return $this
     */
    public function prepareForValidation(): self
    {
        $this->getValidatorInstance();

        return $this;
    }

    /**
     * Set the query on the request.
     *
     * @param array<int|string, mixed> $query
     * @return $this
     */
    public function query(array $query): self
    {
        $this->baseRequest->query = new InputBag($query);

        return $this;
    }

    /**
     * Set a route parameter on the request.
     *
     * @param string $key
     * @param string|object|null $value
     * @return $this
     */
    public function route(string $key, string|object|null $value): self
    {
        $route = $this->baseRequest->route();
        /** @var \Illuminate\Routing\Route $route */
        $route->setParameter($key, $value);

        return $this;
    }
}
