<?php

namespace Recoded\TestingTools\PHPUnit\Constraints;

use Illuminate\Database\Eloquent\Model;
use PHPUnit\Framework\Constraint\Constraint;
use SebastianBergmann\Comparator\ComparisonFailure;

/**
 * @internal
 */
class IsModel extends Constraint
{
    /**
     * The expected model.
     *
     * @var \Illuminate\Database\Eloquent\Model
     */
    protected Model $value;

    /**
     * Create a new constraint instance.
     *
     * @param \Illuminate\Database\Eloquent\Model $value
     * @return void
     */
    public function __construct(Model $value)
    {
        $this->value = $value;
    }

    /**
     * Determine whether the failure was caused by the connections.
     *
     * @param \Illuminate\Database\Eloquent\Model $other
     * @return bool
     */
    protected function causedByConnection(Model $other): bool
    {
        return $other->getConnectionName() !== $this->value->getConnectionName();
    }

    /**
     * Determine whether the failure was caused by the keys.
     *
     * @param \Illuminate\Database\Eloquent\Model $other
     * @return bool
     */
    protected function causedByKey(Model $other): bool
    {
        return $other->getKey() !== $this->value->getKey();
    }

    /**
     * Determine whether the failure was caused by the tables.
     *
     * @param \Illuminate\Database\Eloquent\Model $other
     * @return bool
     */
    protected function causedByTable(Model $other): bool
    {
        return $other->getTable() !== $this->value->getTable();
    }

    /**
     * Evaluate the constraint.
     *
     * @param mixed $other
     * @param string $description
     * @param bool $returnResult
     * @return bool|null
     */
    public function evaluate(mixed $other, string $description = '', bool $returnResult = false): ?bool
    {
        $success = $this->matches($other);

        if ($returnResult) {
            return $success;
        }

        if (!$success) {
            $comparisonFailure = null;

            if ($other instanceof Model) {
                if ($this->causedByConnection($other)) {
                    $comparisonFailure = new ComparisonFailure(
                        $expected = $this->value->getConnectionName(),
                        $actual = $other->getConnectionName(),
                        $this->exporter()->export($expected),
                        $this->exporter()->export($actual),
                    );
                } elseif ($this->causedByTable($other)) {
                    $comparisonFailure = new ComparisonFailure(
                        $expected = $this->value->getTable(),
                        $actual = $other->getTable(),
                        $this->exporter()->export($expected),
                        $this->exporter()->export($actual),
                    );
                } elseif ($this->causedByKey($other)) {
                    $comparisonFailure = new ComparisonFailure(
                        $expected = $this->value->getKey(),
                        $actual = $other->getKey(),
                        $this->exporter()->export($expected),
                        $this->exporter()->export($actual),
                    );
                }
            }

            $this->fail($other, $description, $comparisonFailure);
        }

        return null;
    }

    /**
     * Get the description of the failure.
     *
     * @param mixed $other
     * @return string
     */
    protected function failureDescription(mixed $other): string
    {
        $additional = '';

        if ($other instanceof Model) {
            $otherAsString = 'the given model';

            if ($this->causedByConnection($other)) {
                $additional = ', the connections do not match';
            } elseif ($this->causedByTable($other)) {
                $additional = ', the tables do not match';
            } elseif ($this->causedByKey($other)) {
                $additional = ', the keys do not match';
            }
        } elseif (is_object($other)) {
            $otherAsString = sprintf('an object of class "%s"', get_class($other));
        } else {
            $otherAsString = $this->exporter()->export($other);
        }

        return $otherAsString . ' ' . $this->toString() . $additional;
    }

    /**
     * Check if the given model is the expected one.
     *
     * @param mixed $other
     * @return bool
     */
    public function matches($other): bool
    {
        return $other instanceof Model && $this->value->is($other);
    }

    /**
     * Get a string representation of the constraint.
     *
     * @return string
     */
    public function toString(): string
    {
        return 'is the expected model';
    }
}
